import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/cards/card1.dart';
import 'package:wordpress_app/cards/card4.dart';
import 'package:wordpress_app/config/config.dart';
import 'package:wordpress_app/utils/empty_image.dart';
import 'package:wordpress_app/utils/loading_card.dart';
import 'package:wordpress_app/widgets/inline_ads.dart';
import 'package:wordpress_app/widgets/scroll_to_top_button.dart';
import '../models/article.dart';
import 'package:easy_localization/easy_localization.dart';
import 'dart:async';

import '../services/wordpress_service.dart';

class PopularArticlesPage extends ConsumerStatefulWidget {
  const PopularArticlesPage({super.key});

  @override
  ConsumerState<PopularArticlesPage> createState() =>
      _PopularArticlesPageState();
}

class _PopularArticlesPageState extends ConsumerState<PopularArticlesPage> {
  final List<Article> _articles = [];
  bool? _hasData;
  var scaffoldKey = GlobalKey<ScaffoldState>();
  final String _timeRange = 'last30days';
  final int _postLimit = 30;
  final ScrollController _scrollController = ScrollController();

  @override
  void initState() {
    _fetchArticles();
    _hasData = true;
    super.initState();
  }

  @override
  void dispose() {
    _scrollController.dispose();
    super.dispose();
  }

  Future _fetchArticles() async {
    await WordPressService()
        .fetchPopularPosts(_timeRange, _postLimit)
        .then((value) {
      _articles.addAll(value);
      if (_articles.isEmpty) {
        _hasData = false;
      }
      setState(() {});
    });
  }

  Future _onRefresh() async {
    setState(() {
      _hasData = true;
      _articles.clear();
    });
    await _fetchArticles();
  }

  @override
  Widget build(BuildContext context) {
    final configs = ref.read(configBlocProvider).configs!;
    return Scaffold(
      body: RefreshIndicator(
        child: Stack(
          children: [
            CustomScrollView(
              controller: _scrollController,
              slivers: <Widget>[
                SliverAppBar(
                  automaticallyImplyLeading: false,
                  pinned: true,
                  leading: const BackButton(color: Colors.white),
                  backgroundColor: Theme.of(context).primaryColor,
                  expandedHeight: MediaQuery.of(context).size.height * 0.15,
                  flexibleSpace: FlexibleSpaceBar(
                    centerTitle: false,
                    background: Container(
                      color: Theme.of(context).primaryColor,
                      width: double.infinity,
                    ),
                    title: const Text('popular-contents',
                            style: TextStyle(color: Colors.white))
                        .tr(),
                  ),
                ),
                _hasData == false
                    ? SliverFillRemaining(
                        child: Column(
                        children: [
                          SizedBox(
                            height: MediaQuery.of(context).size.height * 0.20,
                          ),
                          EmptyPageWithImage(
                              image: Config.noContentImage,
                              title: 'no-contents'.tr()),
                        ],
                      ))
                    : _articles.isEmpty && _hasData == true
                        ? SliverPadding(
                            padding: const EdgeInsets.all(20),
                            sliver: SliverList.separated(
                              itemCount: 6,
                              itemBuilder: (context, index) {
                                return const LoadingCard(height: 250);
                              },
                              separatorBuilder: (context, index) {
                                return const Divider(height: 40);
                              },
                            ),
                          )
                        : SliverPadding(
                            padding: const EdgeInsets.all(20),
                            sliver: SliverList.separated(
                              itemCount: _articles.length,
                              itemBuilder: (context, index) {
                                if ((index + 1) % configs.postIntervalCount ==
                                    0) {
                                  return Column(
                                    children: [
                                      Card1(
                                          article: _articles[index],
                                          heroTag:
                                              'popular_page_${_articles[index].id}'),
                                      const Divider(height: 60),
                                      const InlineAds(isSliver: true),
                                    ],
                                  );
                                } else {
                                  return Card4(
                                      article: _articles[index],
                                      heroTag:
                                          'popular_page1_${_articles[index].id}');
                                }
                              },
                              separatorBuilder: (context, index) {
                                return const Divider(height: 40);
                              },
                            ),
                          )
              ],
            ),
            Positioned(
              bottom: 50,
              right: 16,
              child: ScrollToTopButton(controller: _scrollController),
            ),
          ],
        ),
        onRefresh: () async => _onRefresh(),
      ),
    );
  }
}
